package nemosofts.voxradio.activity;

import android.app.Application;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.view.Window;
import android.widget.ProgressBar;
import android.widget.TextView;
import android.widget.Toast;

import androidx.activity.EdgeToEdge;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.view.WindowCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.core.view.WindowInsetsControllerCompat;
import androidx.nemosofts.LauncherListener;
import androidx.nemosofts.LauncherTask;
import androidx.nemosofts.theme.ColorUtils;
import androidx.nemosofts.utils.NetworkUtils;

import com.google.firebase.auth.FirebaseAuth;
import com.google.firebase.auth.FirebaseUser;

import nemosofts.voxradio.BuildConfig;
import nemosofts.voxradio.R;
import nemosofts.voxradio.callback.Callback;
import nemosofts.voxradio.callback.Method;
import nemosofts.voxradio.executor.LoadAbout;
import nemosofts.voxradio.executor.LoadLogin;
import nemosofts.voxradio.interfaces.AboutListener;
import nemosofts.voxradio.interfaces.LoginListener;
import nemosofts.voxradio.utils.ApplicationUtil;
import nemosofts.voxradio.utils.IfSupported;
import nemosofts.voxradio.utils.helper.DBHelper;
import nemosofts.voxradio.utils.helper.Helper;
import nemosofts.voxradio.utils.helper.SPHelper;
import nemosofts.voxradio.utils.purchases.BillingUpdate;

public class LauncherActivity extends AppCompatActivity implements LauncherListener {

    private Helper helper;
    private SPHelper spHelper;
    private DBHelper dbHelper;
    private ProgressBar pb;
    Application application;

    private BillingUpdate billingUpdate;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        IfSupported.isRTL(this);
        IfSupported.isScreenshot(this);
        EdgeToEdge.enable(this);
        setContentView(R.layout.activity_launcher);
        hideNavigationBarStatusBars();

        helper = new Helper(this);
        spHelper = new SPHelper(this);
        dbHelper = new DBHelper(this);

        pb = findViewById(R.id.pb_splash);

        findViewById(R.id.rl_splash).setBackgroundColor(ColorUtils.colorBg(this));

        TextView appName = findViewById(R.id.tv_app_name);
        appName.setTextColor(ColorUtils.colorTitle(this));

        TextView appVersion = findViewById(R.id.tv_version);
        appVersion.setTextColor(ColorUtils.colorTitleSub(this));
        String version = getString(R.string.version) + " " + BuildConfig.VERSION_NAME;
        appVersion.setText(version);

        billingUpdate = new BillingUpdate(LauncherActivity.this, new BillingUpdate.Listener() {
            @Override
            public void onBillingServiceDisconnected() {
                if (isFinishing()){
                    return;
                }
                loadAboutData();
            }

            @Override
            public void onBillingSetupFinished(boolean isSubscribed) {
                if (isFinishing()){
                    return;
                }
                spHelper.setIsSubscribed(isSubscribed);
                loadAboutData();
            }
        });
    }

    private void loadAboutData() {
        if (!NetworkUtils.isConnected(this)) {
            if (Boolean.TRUE.equals(spHelper.getIsFirst())) {
                errorDialog(getString(R.string.err_internet_not_connected), getString(R.string.err_try_internet_connected));
                return;
            }
            try {
                dbHelper.getAbout();
                setSaveData();
            } catch (Exception e) {
                errorDialog(getString(R.string.err_internet_not_connected), getString(R.string.err_try_internet_connected));
            }
            return;
        }

        LoadAbout loadAbout = new LoadAbout(LauncherActivity.this, new AboutListener() {
            @Override
            public void onStart() {
                pb.setVisibility(View.VISIBLE);
            }

            @Override
            public void onEnd(String success, String verifyStatus, String message){
                if (isFinishing()){
                    return;
                }
                pb.setVisibility(View.GONE);
                if (success.equals("1")){
                    if (!verifyStatus.equals("-1") && !verifyStatus.equals("-2")){
                        dbHelper.addToAbout();
                        setSaveData();
                    } else {
                        errorDialog(getString(R.string.err_unauthorized_access), message);
                    }
                } else {
                    errorDialog(getString(R.string.err_server), getString(R.string.err_server_not_connected));
                }
            }
        });
        loadAbout.execute();
    }

    private void setSaveData() {
        new LauncherTask(this, this).execute();
    }

    private void loadSettings() {
        if (Boolean.TRUE.equals(Callback.getIsAppUpdate()) && Callback.getAppNewVersion() != BuildConfig.VERSION_CODE){
            openDialogActivity(Callback.DIALOG_TYPE_UPDATE);
        } else if(Boolean.TRUE.equals(spHelper.getIsMaintenance())){
            openDialogActivity(Callback.DIALOG_TYPE_MAINTENANCE);
        } else {
            if (Boolean.TRUE.equals(spHelper.getIsFirst())) {
                if (Boolean.TRUE.equals(spHelper.getIsLogin())){
                    openSignInActivity();
                } else {
                    spHelper.setIsFirst(false);

                    application = getApplication();
                    ((MyApplication) application).loadAd(LauncherActivity.this);

                    openMainActivity();
                }
            } else {
                loadActivity();
            }
        }
    }

    private void loadActivity() {
        application = getApplication();
        ((MyApplication) application).loadAd(LauncherActivity.this);

        if (Boolean.FALSE.equals(spHelper.getIsAutoLogin())) {
            openMainActivity();
        } else {
            if (spHelper.getLoginType().equals(Method.LOGIN_TYPE_GOOGLE)) {
                FirebaseUser currentUser = FirebaseAuth.getInstance().getCurrentUser();
                if (currentUser != null) {
                    loadLogin(Method.LOGIN_TYPE_GOOGLE, spHelper.getAuthID());
                } else {
                    spHelper.setIsAutoLogin(false);
                    openMainActivity();
                }
            } else {
                loadLogin(Method.LOGIN_TYPE_NORMAL, "");
            }
        }
    }

    private void loadLogin(final String loginType, final String authID) {
        if (!NetworkUtils.isConnected(this)) {
            Toast.makeText(LauncherActivity.this, getString(R.string.err_internet_not_connected), Toast.LENGTH_SHORT).show();
            spHelper.setIsAutoLogin(false);
            openMainActivity();
            return;
        }
        LoadLogin loadLogin = new LoadLogin(new LoginListener() {
            @Override
            public void onStart() {
                pb.setVisibility(View.VISIBLE);
            }

            @Override
            public void onEnd(String success, String loginSuccess, String message, String userID,
                              String userName, String userGender, String userPhone, String profile) {
                if (isFinishing()){
                    return;
                }
                pb.setVisibility(View.GONE);
                if (success.equals("1") && (!loginSuccess.equals("-1"))) {
                    spHelper.setLoginDetails(userID, userName, userPhone, spHelper.getEmail(), userGender,
                            profile, authID, spHelper.getIsRemember(), spHelper.getPassword(), loginType
                    );
                    spHelper.setIsLogged(true);
                }
                openMainActivity();
            }
        }, helper.getAPIRequest(Method.METHOD_LOGIN, 0,"","","",
                "","","",spHelper.getEmail(),"","",
                spHelper.getPassword(),authID,loginType,null));
        loadLogin.execute();
    }

    private void openMainActivity() {
        pb.setVisibility(View.VISIBLE);
        new Handler(Looper.getMainLooper()).postDelayed(() -> {
            if (isFinishing()){
                return;
            }
            Intent intent;
            if (!((MyApplication) application).getAppOpenAdManager().isAdAvailable()
                    || (Callback.getIsAppOpenAdShown()
                    && !((MyApplication) application).getAppOpenAdManager().getIsShowingAd())) {
                intent = new Intent(LauncherActivity.this, MainActivity.class);
                intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
                startActivity(intent);
                finish();
            } else if(((MyApplication) application).getAppOpenAdManager().isAdAvailable()
                    && Boolean.TRUE.equals(!((MyApplication) application).getAppOpenAdManager().getIsShowingAd())) {
                ((MyApplication) application).getAppOpenAdManager().showAdIfAvailable(LauncherActivity.this);
            }
        }, 5500);
    }

    private void openSignInActivity() {
        Intent intent = new Intent(LauncherActivity.this, SignInActivity.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
        intent.putExtra("from", "");
        startActivity(intent);
        finish();
    }

    private void openDialogActivity(String type) {
        Intent intent = new Intent(LauncherActivity.this, DialogActivity.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
        intent.putExtra("from", type);
        startActivity(intent);
        finish();
    }

    @Override
    public void onStartPairing() {
        pb.setVisibility(View.VISIBLE);
    }

    @Override
    public void onConnected() {
        pb.setVisibility(View.GONE);
        loadSettings();
    }

    @Override
    public void onError(String message) {
        pb.setVisibility(View.GONE);
        if (message == null || message.isEmpty()){
            errorDialog(getString(R.string.err_server), "Server not connected lib");
        } else {
            errorDialog(getString(R.string.err_unauthorized_access), message);
        }
    }

    private void errorDialog(String title, String message) {
        final AlertDialog.Builder alertDialog = new AlertDialog.Builder(LauncherActivity.this, R.style.dialogTheme);
        alertDialog.setTitle(title);
        alertDialog.setMessage(message);
        alertDialog.setCancelable(false);
        if (title.equals(getString(R.string.err_internet_not_connected)) || title.equals(getString(R.string.err_server_not_connected))) {
            alertDialog.setNegativeButton(getString(R.string.retry), (dialog, which) -> loadAboutData());
        }
        alertDialog.setPositiveButton(getString(R.string.exit), (dialog, which) -> finish());
        alertDialog.show();
    }

    @Override
    public void onResume() {
        super.onResume();
        if (billingUpdate != null){
            billingUpdate.resume();
        }
    }

    @Override
    public void onDestroy() {
        try {
            if (dbHelper != null){
                dbHelper.close();
            }
            if (billingUpdate != null) {
                billingUpdate.release();
            }
        } catch (Exception e) {
            ApplicationUtil.log("LauncherActivity", "Error in closing", e);
        }
        super.onDestroy();
    }

    public void hideNavigationBarStatusBars() {
        try {
            Window window = getWindow();
            View decorView = window.getDecorView();

            // Allow content to extend behind system bars
            WindowCompat.setDecorFitsSystemWindows(window, false);

            // Use compatible insets controller
            WindowInsetsControllerCompat controller = new WindowInsetsControllerCompat(window, decorView);

            // Hide status and navigation bars
            controller.hide(WindowInsetsCompat.Type.statusBars() | WindowInsetsCompat.Type.navigationBars());

            // Enable immersive sticky behavior (swipe to temporarily show bars)
            controller.setSystemBarsBehavior(
                    WindowInsetsControllerCompat.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
            );
        } catch (Exception e) {
            ApplicationUtil.log("LauncherActivity", "Failed to hide Navigation Bar & Status Bar", e);
        }
    }
}